<?php

namespace App\Http\Controllers\Api\Drivers;

use App\Events\UpdateShipmentEvent;
use App\Http\Controllers\Controller;
use App\Http\Resources\DriverShipmentResource;
use App\Http\Resources\InsuranceTypeResource;
use App\Http\Resources\ShipmentRequestResource;
use App\Http\Resources\TruckCategoryResource;
use App\Http\Resources\TruckHeightResource;
use App\Http\Resources\TruckLengthResource;
use App\Http\Resources\TruckLoadCapacityResource;
use App\Http\Resources\TruckModelResource;
use App\Http\Resources\TruckNumberOfAxlesResource;
use App\Http\Resources\TruckTypeResource;
use App\Http\Resources\TruckWidthResource;
use App\Models\AppSetting;
use App\Models\Driver;
use App\Models\FromAddress;
use App\Models\FromAddressDetail;
use App\Models\InsuranceType;
use App\Models\Shipment;
use App\Models\ShipmentDriver;
use App\Models\ToAddress;
use App\Models\ToAddressDetails;
use App\Models\TruckCategory;
use App\Models\TruckHeight;
use App\Models\TruckLength;
use App\Models\TruckLoadCapacity;
use App\Models\TruckModel;
use App\Models\TruckNumberOfAxles;
use App\Models\TruckType;
use App\Models\TruckWidth;
use App\Models\User;
use App\Services\External\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class DriverShipmentController extends Controller
{

    public function get_shipment_data()
    {
        $types = TruckType::get();
        $models = TruckModel::get();
        $categories = TruckCategory::get();
        $heights = TruckHeight::get();
        $widths = TruckWidth::get();
        $lengths = TruckLength::get();
        $loads = TruckLoadCapacity::get();
        $axles = TruckNumberOfAxles::get();
        $insurance = InsuranceType::get();

        return apiResponse(200, null, [
            'truck_types' => TruckTypeResource::collection($types),
            'truck_models' => TruckModelResource::collection($models),
            'truck_categories' => TruckCategoryResource::collection($categories),
            'truck_heights' => TruckHeightResource::collection($heights),
            'truck_width' => TruckWidthResource::collection($widths),
            'lengths' => TruckLengthResource::collection($lengths),
            'truck_load_capacity' => TruckLoadCapacityResource::collection($loads),
            'truck_number_of_axles' => TruckNumberOfAxlesResource::collection($axles),
            'insurance_types' => InsuranceTypeResource::collection($insurance),
        ]);
    }

    public function list_shipments()
    {
        $driver = auth('driver')->user();
        if ($driver) {
            $shipments = $driver->shipments()->wherePivot('status', 0)->orderBy('shipments.created_at', 'desc')->get();
            return apiResponse(200, 'shipment list', DriverShipmentResource::collection($shipments));
        }
        return apiResponse(401, null, null);

    }

    public function current_shipments()
    {
        $driver = auth('driver')->user();

        $shipments = $driver->shipments()
            ->whereNotIn('shipments.status', [0, 4])
            ->orderBy('created_at', 'desc')
            ->get();

        return apiResponse(200, null, DriverShipmentResource::collection($shipments));
    }

    public function previous_shipments()
    {
        $driver = auth('driver')->user();

        $shipments = $driver->shipments()
            ->where('shipments.status', 4)
            ->orderBy('created_at', 'desc')
            ->get();

        return apiResponse(200, null, DriverShipmentResource::collection($shipments));
    }

    public function shipment_requests()
    {
        $driver = auth('driver')->user();
        $shipments = Shipment::whereDoesntHave('drivers')->get();
        return apiResponse(200, null, ShipmentRequestResource::collection($shipments));
    }

    public function accept_price(Request $request)
    {
        $shipment = Shipment::findOrFail($request->shipment_id);
        $driverId = auth('driver')->user()->id;
        $shipment->drivers()->detach($driverId);
        $appPrice = $shipment->user_price * (AppSetting::key('app_percentage')->first()->value / 100);
        $taxPrice = $shipment->user_price * (AppSetting::key('tax')->first()->value / 100);

        $shipment->drivers()->attach($driverId, ['status' => 1, 'price' => $shipment->user_price, 'app_price' => $appPrice, 'tax' => $taxPrice]);

        $shipment->update([
            'status' => 5, // accepted but not start packing
        ]);
        $shipment->final_price = $shipment->user_price;

        $shipment->save();
        $drivers = Driver::whereIn('id', ShipmentDriver::where('shipment_id', $request->shipment_id)->where('status', 1)->pluck('driver_id'))->get();
        Storage::disk('local')->put('a7maaa', 'str: ' . json_encode(auth('driver')->user()));

        broadcast(new UpdateShipmentEvent($shipment, auth('driver')->user(), count($drivers)));

        Notification::notify('users', 'shipments', [$shipment->user_id], $shipment->id, 'new-shipment', 'new-shipment');

        //        return  $shipment;

        return apiResponse(200, 'shipment is accepted', $shipment);
    }

    public function offer_higher_price(Request $request)
    {
        $shipment = Shipment::findOrFail($request->shipment_id);
        $appPrice = $request->price * (AppSetting::key('app_percentage')->first()->value / 100);
        $taxPrice = $request->price * (AppSetting::key('tax')->first()->value / 100);
        $shipment->drivers()->updateExistingPivot(auth('driver')->user()->id, ['price' => $request->price, 'app_price' => $appPrice, 'tax' => $taxPrice]);
        $shipment->update(['driver_price' => $request->price]);
        broadcast(new UpdateShipmentEvent($shipment, auth('driver')->user()));

        return apiResponse(200, 'driver price is updated', $request->price);
    }

    public function choice_drivers(Request $request)
    {
        $shipment = Shipment::findOrFail($request->shipment_id);
        $shipment_driver = ShipmentDriver::where('status', 0)->where('driver_id', $request->driver_id)->where('shipment_id', $request->shipment_id)->first();
        $shipment_driver->status = 1;
        $shipment_driver->save();

        return apiResponse(200, null, $shipment_driver);
    }

    public function reject_driver(Request $request)
    {
        $shipment = Shipment::findOrFail($request->shipment_id);
        $shipment->drivers()->detach($request->driver_id);
        return apiResponse(200, 'driver price is rejected', null);
    }

    public function change_status(Request $request)
    {
        $shipment = Shipment::where('id', $request->shipment_id)->first();
        $shipment->status = $request->status;
        $shipment->save();
        return apiResponse(200, 'status is changed', null);
    }

    public function stop_requests()
    {
        try {
            $driver = auth('driver')->user();

            if ($driver->stop_receiving_shipment_requests == 0) {
                $driver->stop_receiving_shipment_requests = 1;
                $driver->save();

            } else {
                $driver->stop_receiving_shipment_requests = 0;
                $driver->save();
            }

            return apiResponse(200, $driver->stop_receiving_shipment_requests == 1 ? 'stop receiving shipment requests is on'
                : 'stop receiving shipment requests is off', $driver->stop_receiving_shipment_requests);

        } catch (\Throwable $th) {
            return apiResponse($th->getCode(), $th->getMessage(), null);
        }
    }

    public function stop_requests_status()
    {
        return apiResponse(200, null, auth('driver')->user()->stop_receiving_shipment_requests);

    }

    public function search(Request $request)
    {
        try {
            $query = Shipment::query();

            if ($request->keyword) {
                $keyword = $request->keyword;
                $shipments = $query->where('user_price', 'like', '%' . $keyword . '%')
                    ->orWhere('description', 'like', '%' . $keyword . '%')
                    ->orWhere('type', 'like', '%' . $keyword . '%')
                    ->orWhereHas('receiver', function ($item) use ($keyword) {
                        $item->where('name', 'like', '%' . $keyword . '%');
                    })

                    ->get();

                if (count($shipments)) {
                    return apiResponse(200, null, DriverShipmentResource::collection($shipments));
                }
                return apiResponse(404, 'no results', null);
            }
            return apiResponse(404, 'keyword not found', null);

        } catch (\Throwable $th) {
            return apiResponse($th->getCode(), $th->getMessage(), null);
        }
    }

    public function get_shipment_details(Request $request)
    {
        $shipment_details = Shipment::where('id', $request->shipment_id)->first();

        $shipment_details->from_address = FromAddress::select('latitude', 'longitude')->where('id', $shipment_details->from_address_id)->first();

        $from_address_details = FromAddressDetail::with('governorate', 'country')->select('address_name', 'street_name', 'description', 'governorate_id', 'country_id')->where('id', $shipment_details->from_address_detail_id)->first();

        if ($shipment_details->from_address) {
            $shipment_details->from_address = [
                'latitude' => $shipment_details->from_address->latitude,
                'longitude' => $shipment_details->from_address->longitude,
                'address_name' => $from_address_details->address_name ?? "",
                'street_name' => $from_address_details->street_name ?? "",
                'description' => $from_address_details->description ?? "",
                'governorate_name' => $from_address_details->governorate->name_ar ?? "",
                'country_name' => $from_address_details->country->name_ar ?? "",
            ];
        } else {
            $shipment_details->from_address = [
                'latitude' => "",
                'longitude' => "",
                'address_name' => "",
                'street_name' => "",
                'description' => "",
                'governorate_name' => "",
                'country_name' => "",
            ];
        }

        $shipment_details->to_address = ToAddress::select('latitude', 'longitude')->where('id', $shipment_details->to_address_id)->first();

        $to_address_details = ToAddressDetails::with('governorate', 'country')->select('address_name', 'street_name', 'description', 'governorate_id', 'country_id')->where('id', $shipment_details->to_address_detail_id)->first();

        $shipment_details->to_address->address_name = $to_address_details->address_name;

        $shipment_details->to_address->street_name = $to_address_details->street_name;

        $shipment_details->to_address->description = $to_address_details->description;

        $shipment_details->to_address->governorate_name = $to_address_details->governorate->name_ar;

        $shipment_details->to_address->country_name = $to_address_details->country->name_ar;

        $shipment_details->drivers = ShipmentDriver::where('shipment_id', $shipment_details->id)->where('status', 1)->get();

        $all_drivers = array();

        foreach ($shipment_details->drivers as $single_driver) {

            $driver_temp = Driver::select('id', 'name', 'profile_image', 'mobile')->where('id', $single_driver->driver_id)->first();
            $driver_temp->profile_image = asset('images/') . '/' . $driver_temp->profile_image;
            array_push($all_drivers, $driver_temp);
        }
        $shipment_details->drivers = $all_drivers;

        $shipment_details->sender_details = User::select('id', 'name', 'image', 'mobile', 'rate')->where('id', $shipment_details->sender_id)->first();

        $shipment_details->receiver_details = User::select('id', 'name', 'image', 'mobile', 'rate')->where('id', $shipment_details->receiver_id)->first();

        //        $shipment_details->shipment_invoice_picture =  $shipment_details->shipment_invoice_picture;
//        $shipment_details->packing_list_picture =  $shipment_details->packing_list_picture;
//        $shipment_details->sender_details->image = asset('images/').'/'. $shipment_details->sender_details->image;
//        $shipment_details->receiver_details->image = asset('images/').'/'. $shipment_details->receiver_details->image;
        $insurance = InsuranceType::where('id', $shipment_details->insurance_type_id)->first();

        $shipment_details->insurance_type = $insurance->type;

        $shipment_details->insurance_price = $insurance->price;


        $insuranceTypePrice = InsuranceType::whereId($shipment_details->insurance_type_id)->first()->price;
        $shipmentPrice = $shipment_details->distance * $shipment_details->kilo_price;
        $taxPrice = $shipmentPrice * (AppSetting::key('tax')->first()->value / 100);
        $insuranceFinalPrice = $shipment_details->final_price * ($insuranceTypePrice / 100);
        $shipment_details['tax_price'] = (number_format($taxPrice, 2));
        $shipment_details['insurance_final_price'] = number_format($insuranceFinalPrice, 2);

        return apiResponse(200, null, $shipment_details);
    }
}
